<?php
// Enable CORS to allow frontend requests
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

include 'db.php';

// Response function
function sendResponse($success, $message = '', $data = null) {
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data
    ]);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Invalid request method');
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate input
if (!isset($input['phone_number']) || 
    !isset($input['otp']) || 
    !preg_match('/^\d{10}$/', $input['phone_number']) ||
    !preg_match('/^\d{6}$/', $input['otp'])) {
    sendResponse(false, 'Invalid input parameters');
}

$phoneNumber = $input['phone_number'];
$submittedOtp = $input['otp'];

try {
    

    // Prepare statement to check OTP
    $stmt = $conn->prepare("SELECT otp, expires_at FROM otp_attempts WHERE phone_number = ? AND otp = ? AND expires_at > NOW()");
    $stmt->bind_param("ss", $phoneNumber, $submittedOtp);
    $stmt->execute();
    $result = $stmt->get_result();

    // Check if OTP exists and is not expired
    if ($result->num_rows === 0) {
        // Check if OTP exists but is expired
        $expiredStmt = $conn->prepare("SELECT otp FROM otp_attempts WHERE phone_number = ? AND otp = ?");
        $expiredStmt->bind_param("ss", $phoneNumber, $submittedOtp);
        $expiredStmt->execute();
        $expiredResult = $expiredStmt->get_result();

        if ($expiredResult->num_rows > 0) {
            sendResponse(false, 'OTP has expired. Please request a new OTP.');
        } else {
            sendResponse(false, 'Invalid OTP');
        }
    }

    // OTP is valid, delete the used OTP
    $deleteStmt = $conn->prepare("DELETE FROM otp_attempts WHERE phone_number = ?");
    $deleteStmt->bind_param("s", $phoneNumber);
    $deleteStmt->execute();

    // Close statements and connection
    $stmt->close();
    $deleteStmt->close();
    $conn->close();

    // Send successful verification response
    sendResponse(true, 'OTP verified successfully');

} catch (Exception $e) {
    // Handle any errors
    sendResponse(false, $e->getMessage());
}